﻿using Microsoft.Crm.Sdk.Messages;
using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Messages;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Collections.Generic;
using System.Linq;
using System.ServiceModel;
using System.Text;
using System.Threading.Tasks;
using VA.PPMS.Context;

namespace VA.PPMS.CRM.Plugins.Agreement
{
    public class AgreementUpdate : IPlugin
    {
        private ITracingService _tracingService;
        private const string PluginName = "AgreementUpdate";

        public void Execute(IServiceProvider serviceProvider)
        {
            // Tracing service for debugging
            _tracingService = (ITracingService)serviceProvider.GetService(typeof(ITracingService));

            // Get execution context
            IPluginExecutionContext context = (IPluginExecutionContext)serviceProvider.GetService(typeof(IPluginExecutionContext));

            if (context.InputParameters.Contains("Target") && context.InputParameters["Target"] is Entity)
            {
                _tracingService.Trace("Begin");

                // Obtain the target entity from the input parameters.
                Entity provider = (Entity)context.InputParameters["Target"];

                // Verify target entity type
                if (provider.LogicalName != "ppms_provideragreement")
                    return;

                _tracingService.Trace("Entity found");

                // Get organization service reference
                IOrganizationServiceFactory serviceFactory = (IOrganizationServiceFactory)serviceProvider.GetService(typeof(IOrganizationServiceFactory));
                IOrganizationService service = serviceFactory.CreateOrganizationService(context.UserId);

                try
                {
                    // Get related entities to deactivate
                    _tracingService.Trace("Retrieve Agreement and related entities");
                    Entity entity = GetAgreement(service, provider.Id);
                    if (entity == null)
                    {
                        _tracingService.Trace("Failed to retrieve agreement related entities");
                        return;
                    }
                    _tracingService.Trace("Agreement Retrieved");

                    List<SetStateRequest> requests = new List<SetStateRequest>();

                    // handle event based on message type
                    _tracingService.Trace("Check status");
                    var isActive = IsActive(entity);
                    int state = 0;
                    int statuscode = 0;

                    if (isActive)
                    {
                        _tracingService.Trace("Agreement in Active Status");
                        state = (int)ppms_providerserviceState.Active;
                        statuscode = (int)ppms_providerservice_StatusCode.Active;
                    }
                    else
                    {
                        _tracingService.Trace("Agreement in Inactive Status");
                        state = (int)ppms_providerserviceState.Inactive;
                        statuscode = (int)ppms_providerservice_StatusCode.Inactive;
                    }

                    _tracingService.Trace("Adding Set State Requests to the related entities");

                    // provider service
                    requests.AddRange(PpmsHelper.GetEntityStateRequests(entity, "ppms_provideragreement_ppms_providerservice_ProviderAgreement", state, statuscode));

                    if (requests != null)
                    {
                        _tracingService.Trace("Exectuting Set State Requests");
                        foreach (var request in requests)
                        {
                            service.Execute(request);
                        }
                    }
                }
                catch (FaultException<OrganizationServiceFault> ex)
                {
                    _tracingService.Trace("Fault: {0}", ex.ToString());
                    throw new InvalidPluginExecutionException(String.Format("An error occurred in {0}.", PluginName), ex);
                }
                catch (Exception ex)
                {
                    _tracingService.Trace("Exception: {0}", ex.ToString());
                    throw;
                }
            }
            _tracingService.Trace("Done");
        }

        private Entity GetAgreement(IOrganizationService service, Guid agreementId)
        {
            var request = new RetrieveRequest();

            request.Target = new EntityReference("ppms_provideragreement", agreementId);
            request.ColumnSet = new ColumnSet(new string[] { "ppms_provideragreementid", "ppms_agreementid", "statecode", "statuscode" });
            request.RelatedEntitiesQuery = new RelationshipQueryCollection();

            //Retrieve related entities
            request.RelatedEntitiesQuery.Add(new Relationship("ppms_provideragreement_ppms_providerservice_ProviderAgreement"),
                new QueryExpression("ppms_providerservice")
                {
                    ColumnSet = new ColumnSet("ppms_providerserviceid", "ppms_name")
                }
            );

            //Get response
            _tracingService.Trace("Excecuting request to retrieve Agreement and relationships");
            var response = (RetrieveResponse)service.Execute(request);
            if (response != null)
                return response.Entity;

            return null;
        }

        /// <summary>
        /// Get state of the specified entity
        /// </summary>
        /// <param name="entity">Entity to check</param>
        /// <returns>true if in Active state and Active status code</returns>
        private bool IsActive(Entity entity)
        {
            var statecode = entity.GetAttributeValue<OptionSetValue>("statecode");
            var statuscode = entity.GetAttributeValue<OptionSetValue>("statuscode");
            _tracingService.Trace("");
            return statecode != null && statecode.Value == (int)AccountState.Active ? true : false;
        }
    }
}
